<?php
/**
 * WebView to App module functionality.
 *
 * @since      1.0.0
 * @package    Bundle_WP
 * @subpackage Bundle_WP/modules/webview-to-app
 */

class Bundle_WP_WebView_To_App {

	/**
	 * The ID of this plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 * @var      string    $plugin_name    The ID of this plugin.
	 */
	private $plugin_name;

	/**
	 * The version of this plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 * @var      string    $version    The current version of this plugin.
	 */
	private $version;

	/**
	 * Module options.
	 *
	 * @since    1.0.0
	 * @access   private
	 * @var      array    $options    Module options.
	 */
	private $options;

	/**
	 * Initialize the class and set its properties.
	 *
	 * @since    1.0.0
	 * @param    string    $plugin_name       The name of this plugin.
	 * @param    string    $version    The version of this plugin.
	 */
	public function __construct( $plugin_name, $version ) {
		$this->plugin_name = $plugin_name;
		$this->version = $version;
		$this->options = get_option( 'bundle_wp_webview_to_app_options', array(
			'app_name' => get_bloginfo('name'),
			'app_short_name' => get_bloginfo('name'),
			'app_description' => get_bloginfo('description'),
			'theme_color' => '#000000',
			'background_color' => '#ffffff',
			'display' => 'standalone',
			'orientation' => 'any',
			'enable_pwa' => false,
			'app_icon' => ''
		));

		// Register settings
		add_action( 'admin_init', array( $this, 'register_settings' ) );

		// Initialize PWA functionality
		if (isset($this->options['enable_pwa']) && $this->options['enable_pwa']) {
			// Add meta tags to header for PWA support
			add_action('wp_head', array($this, 'add_meta_tags'));

			// Enqueue scripts for PWA functionality
			add_action('wp_enqueue_scripts', array($this, 'enqueue_scripts'));

			// Create manifest
			add_action('wp_ajax_nopriv_bundlewp_webviewtoapp_manifest', array($this, 'generate_manifest'));
			add_action('wp_ajax_bundlewp_webviewtoapp_manifest', array($this, 'generate_manifest'));
		}
	}

	/**
	 * Register the settings for this module.
	 *
	 * @since    1.0.0
	 */
	public function register_settings() {
		register_setting(
			'bundle_wp_webview_to_app',
			'bundle_wp_webview_to_app_options',
			array( $this, 'sanitize_options' )
		);

		add_settings_section(
			'bundle_wp_webview_to_app_section',
			__('App Settings', 'bundle-wp'),
			array( $this, 'settings_section_callback' ),
			'bundle_wp_webview_to_app'
		);
		
		add_settings_field(
			'app_name',
			__('App Name', 'bundle-wp'),
			array( $this, 'app_name_render' ),
			'bundle_wp_webview_to_app',
			'bundle_wp_webview_to_app_section'
		);
		
		add_settings_field(
			'app_short_name',
			__('App Short Name', 'bundle-wp'),
			array( $this, 'app_short_name_render' ),
			'bundle_wp_webview_to_app',
			'bundle_wp_webview_to_app_section'
		);
		
		add_settings_field(
			'app_description',
			__('App Description', 'bundle-wp'),
			array( $this, 'app_description_render' ),
			'bundle_wp_webview_to_app',
			'bundle_wp_webview_to_app_section'
		);
		
		add_settings_field(
			'app_icon',
			__('App Icon (512x512px recommended)', 'bundle-wp'),
			array( $this, 'app_icon_render' ),
			'bundle_wp_webview_to_app',
			'bundle_wp_webview_to_app_section'
		);
		
		add_settings_field(
			'theme_color',
			__('Theme Color', 'bundle-wp'),
			array( $this, 'theme_color_render' ),
			'bundle_wp_webview_to_app',
			'bundle_wp_webview_to_app_section'
		);
		
		add_settings_field(
			'background_color',
			__('Background Color', 'bundle-wp'),
			array( $this, 'background_color_render' ),
			'bundle_wp_webview_to_app',
			'bundle_wp_webview_to_app_section'
		);
		
		add_settings_field(
			'display',
			__('Display Mode', 'bundle-wp'),
			array( $this, 'display_render' ),
			'bundle_wp_webview_to_app',
			'bundle_wp_webview_to_app_section'
		);
		
		add_settings_field(
			'orientation',
			__('Orientation', 'bundle-wp'),
			array( $this, 'orientation_render' ),
			'bundle_wp_webview_to_app',
			'bundle_wp_webview_to_app_section'
		);
		
		add_settings_field(
			'enable_pwa',
			__('Enable PWA', 'bundle-wp'),
			array( $this, 'enable_pwa_render' ),
			'bundle_wp_webview_to_app',
			'bundle_wp_webview_to_app_section'
		);
	}

	/**
	 * Sanitize the options.
	 *
	 * @since    1.0.0
	 * @param    array    $input    The input options.
	 * @return   array              The sanitized options.
	 */
	public function sanitize_options( $input ) {
		$sanitized = array();
		
		$sanitized['app_name'] = isset($input['app_name']) ? sanitize_text_field($input['app_name']) : get_bloginfo('name');
		$sanitized['app_short_name'] = isset($input['app_short_name']) ? sanitize_text_field($input['app_short_name']) : get_bloginfo('name');
		$sanitized['app_description'] = isset($input['app_description']) ? sanitize_textarea_field($input['app_description']) : get_bloginfo('description');
		$sanitized['app_icon'] = isset($input['app_icon']) ? esc_url_raw($input['app_icon']) : '';
		$sanitized['theme_color'] = isset($input['theme_color']) ? sanitize_hex_color($input['theme_color']) : '#000000';
		$sanitized['background_color'] = isset($input['background_color']) ? sanitize_hex_color($input['background_color']) : '#ffffff';
		$sanitized['display'] = isset($input['display']) && in_array($input['display'], array('fullscreen', 'standalone', 'minimal-ui', 'browser')) ? $input['display'] : 'standalone';
		$sanitized['orientation'] = isset($input['orientation']) && in_array($input['orientation'], array('any', 'portrait', 'landscape')) ? $input['orientation'] : 'any';
		$sanitized['enable_pwa'] = isset($input['enable_pwa']) ? (bool)$input['enable_pwa'] : false;
		
		return $sanitized;
	}

	/**
	 * Settings section callback.
	 */
	public function settings_section_callback() {
		echo __('Configure your app settings below. These settings will be used to create the web app manifest and improve mobile experience.', 'bundle-wp');
	}
	
	/**
	 * App name field.
	 */
	public function app_name_render() {
		?>
		<input type='text' name='bundle_wp_webview_to_app_options[app_name]' value='<?php echo isset($this->options['app_name']) ? esc_attr($this->options['app_name']) : get_bloginfo('name'); ?>' class="regular-text">
		<p class="description"><?php _e('The name of your web application.', 'bundle-wp'); ?></p>
		<?php
	}
	
	/**
	 * App short name field.
	 */
	public function app_short_name_render() {
		?>
		<input type='text' name='bundle_wp_webview_to_app_options[app_short_name]' value='<?php echo isset($this->options['app_short_name']) ? esc_attr($this->options['app_short_name']) : get_bloginfo('name'); ?>' class="regular-text">
		<p class="description"><?php _e('A shorter name for your web application (displayed on the home screen).', 'bundle-wp'); ?></p>
		<?php
	}
	
	/**
	 * App description field.
	 */
	public function app_description_render() {
		?>
		<textarea name='bundle_wp_webview_to_app_options[app_description]' rows='3' cols='50'><?php echo isset($this->options['app_description']) ? esc_textarea($this->options['app_description']) : get_bloginfo('description'); ?></textarea>
		<p class="description"><?php _e('A brief description of your web application.', 'bundle-wp'); ?></p>
		<?php
	}
	
	/**
	 * App icon field.
	 */
	public function app_icon_render() {
		?>
		<div class="icon-upload-container">
			<input type="hidden" name="bundle_wp_webview_to_app_options[app_icon]" id="app_icon_url" value="<?php echo isset($this->options['app_icon']) ? esc_attr($this->options['app_icon']) : ''; ?>">
			<button type="button" class="button" id="upload_app_icon_button"><?php _e('Upload Icon', 'bundle-wp'); ?></button>
			<div id="app_icon_preview" style="margin-top: 10px; max-width: 150px;">
				<?php if (isset($this->options['app_icon']) && !empty($this->options['app_icon'])) : ?>
					<img src="<?php echo esc_url($this->options['app_icon']); ?>" style="max-width: 100%; height: auto;">
				<?php endif; ?>
			</div>
		</div>
		<p class="description"><?php _e('Upload a square icon for your app (512x512px recommended).', 'bundle-wp'); ?></p>
		<script>
		jQuery(document).ready(function($) {
			$('#upload_app_icon_button').click(function(e) {
				e.preventDefault();
				var image = wp.media({
					title: '<?php _e('Upload App Icon', 'bundle-wp'); ?>',
					multiple: false
				}).open().on('select', function() {
					var uploaded_image = image.state().get('selection').first();
					var image_url = uploaded_image.toJSON().url;
					$('#app_icon_url').val(image_url);
					$('#app_icon_preview').html('<img src="' + image_url + '" style="max-width: 100%; height: auto;">');
				});
			});
		});
		</script>
		<?php
	}
	
	/**
	 * Theme color field.
	 */
	public function theme_color_render() {
		?>
		<input type='text' name='bundle_wp_webview_to_app_options[theme_color]' value='<?php echo isset($this->options['theme_color']) ? esc_attr($this->options['theme_color']) : '#000000'; ?>' class="wvta-color-field">
		<p class="description"><?php _e('The theme color for your app. This affects how the OS displays the site (e.g., on the task switcher).', 'bundle-wp'); ?></p>
		<script>
		jQuery(document).ready(function($) {
			$('.wvta-color-field').wpColorPicker();
		});
		</script>
		<?php
	}
	
	/**
	 * Background color field.
	 */
	public function background_color_render() {
		?>
		<input type='text' name='bundle_wp_webview_to_app_options[background_color]' value='<?php echo isset($this->options['background_color']) ? esc_attr($this->options['background_color']) : '#ffffff'; ?>' class="wvta-color-field">
		<p class="description"><?php _e('The background color for your app, used during loading.', 'bundle-wp'); ?></p>
		<?php
	}
	
	/**
	 * Display mode field.
	 */
	public function display_render() {
		$display = isset($this->options['display']) ? $this->options['display'] : 'standalone';
		?>
		<select name='bundle_wp_webview_to_app_options[display]'>
			<option value='fullscreen' <?php selected($display, 'fullscreen'); ?>><?php _e('Fullscreen', 'bundle-wp'); ?></option>
			<option value='standalone' <?php selected($display, 'standalone'); ?>><?php _e('Standalone', 'bundle-wp'); ?></option>
			<option value='minimal-ui' <?php selected($display, 'minimal-ui'); ?>><?php _e('Minimal UI', 'bundle-wp'); ?></option>
			<option value='browser' <?php selected($display, 'browser'); ?>><?php _e('Browser', 'bundle-wp'); ?></option>
		</select>
		<p class="description"><?php _e('How you would like your app to display on the device.', 'bundle-wp'); ?></p>
		<?php
	}
	
	/**
	 * Orientation field.
	 */
	public function orientation_render() {
		$orientation = isset($this->options['orientation']) ? $this->options['orientation'] : 'any';
		?>
		<select name='bundle_wp_webview_to_app_options[orientation]'>
			<option value='any' <?php selected($orientation, 'any'); ?>><?php _e('Any', 'bundle-wp'); ?></option>
			<option value='portrait' <?php selected($orientation, 'portrait'); ?>><?php _e('Portrait', 'bundle-wp'); ?></option>
			<option value='landscape' <?php selected($orientation, 'landscape'); ?>><?php _e('Landscape', 'bundle-wp'); ?></option>
		</select>
		<p class="description"><?php _e('The default orientation for your app.', 'bundle-wp'); ?></p>
		<?php
	}
	
	/**
	 * Enable PWA field.
	 */
	public function enable_pwa_render() {
		?>
		<input type='checkbox' name='bundle_wp_webview_to_app_options[enable_pwa]' <?php checked(isset($this->options['enable_pwa']) ? $this->options['enable_pwa'] : false, true); ?> value='1'>
		<p class="description"><?php _e('Check this to enable Progressive Web App functionality on your site.', 'bundle-wp'); ?></p>
		<?php
	}

	/**
	 * Display the module admin page.
	 *
	 * @since    1.0.0
	 */
	public function display_admin_page() {
		// Enqueue WordPress media scripts
		wp_enqueue_media();
		// Enqueue WordPress color picker
		wp_enqueue_style('wp-color-picker');
		wp_enqueue_script('wp-color-picker');
		
		require_once BUNDLE_WP_PATH . 'modules/webview-to-app/partials/admin-display.php';
	}

	/**
	 * Add meta tags to header for PWA support.
	 */
	public function add_meta_tags() {
		$options = $this->options;
		
		// Check if PWA is enabled
		if (!isset($options['enable_pwa']) || !$options['enable_pwa']) {
			return;
		}
		
		$theme_color = isset($options['theme_color']) ? $options['theme_color'] : '#000000';
		
		// Meta tags for PWA
		echo '<meta name="viewport" content="width=device-width, initial-scale=1.0, maximum-scale=1.0, user-scalable=no">' . "\n";
		echo '<meta name="theme-color" content="' . esc_attr($theme_color) . '">' . "\n";
		echo '<meta name="mobile-web-app-capable" content="yes">' . "\n";
		echo '<meta name="apple-mobile-web-app-capable" content="yes">' . "\n";
		echo '<meta name="apple-mobile-web-app-status-bar-style" content="black-translucent">' . "\n";
		echo '<meta name="apple-mobile-web-app-title" content="' . esc_attr(isset($options['app_name']) ? $options['app_name'] : get_bloginfo('name')) . '">' . "\n";
		
		// Web app manifest link
		echo '<link rel="manifest" href="' . esc_url(admin_url('admin-ajax.php') . '?action=bundlewp_webviewtoapp_manifest') . '">' . "\n";
		
		// Apple touch icon
		if (isset($options['app_icon']) && !empty($options['app_icon'])) {
			echo '<link rel="apple-touch-icon" href="' . esc_url($options['app_icon']) . '">' . "\n";
			echo '<link rel="icon" sizes="512x512" href="' . esc_url($options['app_icon']) . '">' . "\n";
		}
	}

	/**
	 * Generate web app manifest file.
	 */
	public function generate_manifest() {
		$options = $this->options;
		
		$manifest = array(
			'name' => isset($options['app_name']) ? $options['app_name'] : get_bloginfo('name'),
			'short_name' => isset($options['app_short_name']) ? $options['app_short_name'] : get_bloginfo('name'),
			'description' => isset($options['app_description']) ? $options['app_description'] : get_bloginfo('description'),
			'start_url' => home_url('/?utm_source=pwa'),
			'scope' => home_url('/'),
			'display' => isset($options['display']) ? $options['display'] : 'standalone',
			'orientation' => isset($options['orientation']) ? $options['orientation'] : 'any',
			'background_color' => isset($options['background_color']) ? $options['background_color'] : '#ffffff',
			'theme_color' => isset($options['theme_color']) ? $options['theme_color'] : '#000000',
		);
		
		// Add icons if available
		if (isset($options['app_icon']) && !empty($options['app_icon'])) {
			$manifest['icons'] = array(
				array(
					'src' => $options['app_icon'],
					'sizes' => '512x512',
					'type' => 'image/png',
					'purpose' => 'any maskable'
				)
			);
		}
		
		header('Content-Type: application/json');
		echo json_encode($manifest);
		exit;
	}

	/**
	 * Enqueue scripts for PWA functionality.
	 */
	public function enqueue_scripts() {
		$options = $this->options;
		
		// Check if PWA is enabled
		if (!isset($options['enable_pwa']) || !$options['enable_pwa']) {
			return;
		}
		
		// Register service worker
		wp_enqueue_script(
			'bundle-wp-webview-to-app-register',
			BUNDLE_WP_URL . 'modules/webview-to-app/assets/js/pwa-register.js',
			array(),
			$this->version,
			true
		);
		
		// Add service worker path to JavaScript
		wp_localize_script(
			'bundle-wp-webview-to-app-register',
			'webviewToAppVars',
			array(
				'swPath' => BUNDLE_WP_URL . 'modules/webview-to-app/assets/js/service-worker.js'
			)
		);
	}
}