/**
 * Service Worker for WebView to App module in Bundle WP
 */

const CACHE_NAME = 'bundle-wp-webview-to-app-cache-v1';
const urlsToCache = [
  '/',
  '/wp-includes/css/dist/block-library/style.min.css',
  '/wp-includes/js/jquery/jquery.min.js'
];

// Install event - cache assets
self.addEventListener('install', event => {
  event.waitUntil(
    caches.open(CACHE_NAME)
      .then(cache => {
        console.log('Cache opened');
        return cache.addAll(urlsToCache);
      })
  );
});

// Activate event - cleanup old caches
self.addEventListener('activate', event => {
  event.waitUntil(
    caches.keys().then(cacheNames => {
      return Promise.all(
        cacheNames.map(cacheName => {
          if (cacheName !== CACHE_NAME) {
            console.log('Deleting old cache:', cacheName);
            return caches.delete(cacheName);
          }
        })
      );
    })
  );
});

// Fetch event - serve from cache or network
self.addEventListener('fetch', event => {
  event.respondWith(
    caches.match(event.request)
      .then(response => {
        // Cache hit - return the response from the cached version
        if (response) {
          return response;
        }
        
        // Not in cache - return the result from the live server
        // `fetch` is essentially a "fallback"
        return fetch(event.request)
          .then(response => {
            // Check if we received a valid response
            if (!response || response.status !== 200 || response.type !== 'basic') {
              return response;
            }
            
            // Clone the response
            // We need one to return to the browser and one to save in the cache
            const responseToCache = response.clone();
            
            caches.open(CACHE_NAME)
              .then(cache => {
                // Don't cache API responses or admin pages
                const url = event.request.url;
                if (!url.includes('/wp-admin/') && 
                    !url.includes('/wp-json/') &&
                    !url.includes('/admin-ajax.php')) {
                  cache.put(event.request, responseToCache);
                }
              });
              
            return response;
          });
      })
      .catch(() => {
        // If both cache and network fail, serve an offline fallback
        if (event.request.mode === 'navigate') {
          return caches.match('/');
        }
      })
  );
});

// Push notification event (for future implementation)
self.addEventListener('push', event => {
  if (event.data) {
    const data = event.data.json();
    
    const options = {
      body: data.message,
      icon: data.icon || '/images/notificationicon.jpg',
      badge: data.badge || '/images/badge.jpg'
    };
    
    event.waitUntil(
      self.registration.showNotification(data.title, options)
    );
  }
});

// Notification click event
self.addEventListener('notificationclick', event => {
  event.notification.close();
  
  event.waitUntil(
    clients.openWindow('/')
  );
});