<?php
/**
 * No Search Engine module functionality.
 *
 * @since      1.0.0
 * @package    Bundle_WP
 * @subpackage Bundle_WP/modules/no-search-engine
 */

class Bundle_WP_No_Search_Engine {

	/**
	 * The ID of this plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 * @var      string    $plugin_name    The ID of this plugin.
	 */
	private $plugin_name;

	/**
	 * The version of this plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 * @var      string    $version    The current version of this plugin.
	 */
	private $version;

	/**
	 * Module options.
	 *
	 * @since    1.0.0
	 * @access   private
	 * @var      array    $options    Module options.
	 */
	private $options;

	/**
	 * Initialize the class and set its properties.
	 *
	 * @since    1.0.0
	 * @param    string    $plugin_name       The name of this plugin.
	 * @param    string    $version    The version of this plugin.
	 */
	public function __construct( $plugin_name, $version ) {
		$this->plugin_name = $plugin_name;
		$this->version = $version;
		$this->options = get_option( 'bundle_wp_no_search_engine_options', array(
			'enabled' => false,
			'selected_items' => array(),
			'apply_to' => 'specific_items',
		));

		// Register settings
		add_action( 'admin_init', array( $this, 'register_settings' ) );
	}

	/**
	 * Register the settings for this module.
	 *
	 * @since    1.0.0
	 */
	public function register_settings() {
		register_setting(
			'bundle_wp_no_search_engine',
			'bundle_wp_no_search_engine_options',
			array( $this, 'sanitize_options' )
		);
	}

	/**
	 * Sanitize the options.
	 *
	 * @since    1.0.0
	 * @param    array    $input    The input options.
	 * @return   array              The sanitized options.
	 */
	public function sanitize_options( $input ) {
		$sanitized = array();
		$sanitized['enabled'] = isset( $input['enabled'] ) ? (bool) $input['enabled'] : false;
		$sanitized['apply_to'] = isset( $input['apply_to'] ) && in_array( $input['apply_to'], array( 'entire_site', 'specific_items' ) ) ? sanitize_text_field( $input['apply_to'] ) : 'specific_items';
		$sanitized['selected_items'] = isset( $input['selected_items'] ) && is_array( $input['selected_items'] ) ? array_map( 'intval', $input['selected_items'] ) : array();
		return $sanitized;
	}

	/**
	 * Display the module admin page.
	 *
	 * @since    1.0.0
	 */
	public function display_admin_page() {
		require_once BUNDLE_WP_PATH . 'modules/no-search-engine/partials/admin-display.php';
	}

	/**
	 * Initialize the frontend functionality.
	 *
	 * @since    1.0.0
	 */
	public function init_frontend() {
		if ( $this->should_block_search_engine() ) {
			add_action( 'wp_head', array( $this, 'add_noindex_meta' ), 0 );
			add_filter( 'wp_robots', array( $this, 'filter_robots' ) );
		}
	}

	/**
	 * Check if search engine indexing should be blocked on current page.
	 *
	 * @since    1.0.0
	 * @return   boolean   True if should block, false otherwise.
	 */
	private function should_block_search_engine() {
		// If module is not enabled, don't block
		if ( ! $this->options['enabled'] ) {
			return false;
		}

		// If applying to entire site
		if ( $this->options['apply_to'] == 'entire_site' ) {
			return true;
		}

		// If applying to specific items
		if ( $this->options['apply_to'] == 'specific_items' && ! empty( $this->options['selected_items'] ) ) {
			$post_id = get_the_ID();
			if ( $post_id && in_array( $post_id, $this->options['selected_items'] ) ) {
				return true;
			}
		}

		return false;
	}

	/**
	 * Add noindex meta tag to page header.
	 * For WordPress versions prior to 5.7
	 * 
	 * @since    1.0.0
	 */
	public function add_noindex_meta() {
		// Only needed for WordPress versions before 5.7
		if ( function_exists( 'wp_robots' ) ) {
			return;
		}
		echo '<meta name="robots" content="noindex,nofollow" />' . "\n";
	}
	
	/**
	 * Filter robots for WordPress 5.7+
	 * 
	 * @since    1.0.0
	 * @param    array    $robots    The robots meta array.
	 * @return   array               The filtered robots meta.
	 */
	public function filter_robots( $robots ) {
		$robots['noindex'] = true;
		$robots['nofollow'] = true;
		
		// Additional directives to ensure content is not indexed
		$robots['noimageindex'] = true;
		$robots['noarchive'] = true;
		$robots['nocache'] = true;
		
		return $robots;
	}
}