<?php
/**
 * Block Copy and Paste module functionality.
 *
 * @since      1.0.0
 * @package    Bundle_WP
 * @subpackage Bundle_WP/modules/block-copy-paste
 */

class Bundle_WP_Block_Copy_Paste {

	/**
	 * The ID of this plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 * @var      string    $plugin_name    The ID of this plugin.
	 */
	private $plugin_name;

	/**
	 * The version of this plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 * @var      string    $version    The current version of this plugin.
	 */
	private $version;

	/**
	 * Module options.
	 *
	 * @since    1.0.0
	 * @access   private
	 * @var      array    $options    Module options.
	 */
	private $options;

	/**
	 * Initialize the class and set its properties.
	 *
	 * @since    1.0.0
	 * @param    string    $plugin_name       The name of this plugin.
	 * @param    string    $version    The version of this plugin.
	 */
	public function __construct( $plugin_name, $version ) {
		$this->plugin_name = $plugin_name;
		$this->version = $version;
		$this->options = get_option( 'bundle_wp_block_copy_paste_options', array(
			'enabled' => false,
			'selected_items' => array(),
			'apply_to' => 'entire_site',
		));

		// Register settings
		add_action( 'admin_init', array( $this, 'register_settings' ) );
	}

	/**
	 * Register the settings for this module.
	 *
	 * @since    1.0.0
	 */
	public function register_settings() {
		register_setting(
			'bundle_wp_block_copy_paste',
			'bundle_wp_block_copy_paste_options',
			array( $this, 'sanitize_options' )
		);
	}

	/**
	 * Sanitize the options.
	 *
	 * @since    1.0.0
	 * @param    array    $input    The input options.
	 * @return   array              The sanitized options.
	 */
	public function sanitize_options( $input ) {
		$sanitized = array();
		$sanitized['enabled'] = isset( $input['enabled'] ) ? (bool) $input['enabled'] : false;
		$sanitized['apply_to'] = isset( $input['apply_to'] ) && in_array( $input['apply_to'], array( 'entire_site', 'specific_items' ) ) ? sanitize_text_field( $input['apply_to'] ) : 'entire_site';
		$sanitized['selected_items'] = isset( $input['selected_items'] ) && is_array( $input['selected_items'] ) ? array_map( 'intval', $input['selected_items'] ) : array();
		return $sanitized;
	}

	/**
	 * Display the module admin page.
	 *
	 * @since    1.0.0
	 */
	public function display_admin_page() {
		require_once BUNDLE_WP_PATH . 'modules/block-copy-paste/partials/admin-display.php';
	}

	/**
	 * Initialize the frontend functionality.
	 *
	 * @since    1.0.0
	 */
	public function init_frontend() {
		if ( $this->should_block_copy_paste() ) {
			add_action( 'wp_enqueue_scripts', array( $this, 'enqueue_scripts' ) );
		}
	}

	/**
	 * Check if copy paste should be blocked on current page.
	 *
	 * @since    1.0.0
	 * @return   boolean   True if should block, false otherwise.
	 */
	private function should_block_copy_paste() {
		// If module is not enabled or user is admin, don't block
		if ( ! $this->options['enabled'] || current_user_can( 'manage_options' ) ) {
			return false;
		}

		// If applying to entire site
		if ( $this->options['apply_to'] == 'entire_site' ) {
			return true;
		}

		// If applying to specific items
		if ( $this->options['apply_to'] == 'specific_items' && ! empty( $this->options['selected_items'] ) ) {
			$post_id = get_the_ID();
			if ( $post_id && in_array( $post_id, $this->options['selected_items'] ) ) {
				return true;
			}
		}

		return false;
	}

	/**
	 * Enqueue scripts for the frontend.
	 *
	 * @since    1.0.0
	 */
	public function enqueue_scripts() {
		wp_enqueue_script( $this->plugin_name . '-block-copy-paste', BUNDLE_WP_URL . 'modules/block-copy-paste/js/bundle-wp-block-copy-paste.js', array( 'jquery' ), $this->version, true );
	}
}