<?php
/**
 * The admin-specific functionality of the plugin.
 *
 * @since      1.0.0
 * @package    Bundle_WP
 */

class Bundle_WP_Admin {

	/**
	 * The ID of this plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 * @var      string    $plugin_name    The ID of this plugin.
	 */
	private $plugin_name;

	/**
	 * The version of this plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 * @var      string    $version    The current version of this plugin.
	 */
	private $version;

	/**
	 * Initialize the class and set its properties.
	 *
	 * @since    1.0.0
	 * @param    string    $plugin_name       The name of this plugin.
	 * @param    string    $version    The version of this plugin.
	 */
	public function __construct( $plugin_name, $version ) {
		$this->plugin_name = $plugin_name;
		$this->version = $version;
		
		// Initialize module properties
		$this->block_copy_paste = null;
		$this->no_search_engine = null;
		$this->webview_to_app = null;
	}

	/**
	 * Register the stylesheets for the admin area.
	 *
	 * @since    1.0.0
	 */
	public function enqueue_styles() {
		wp_enqueue_style( $this->plugin_name, BUNDLE_WP_URL . 'admin/css/bundle-wp-admin.css', array(), $this->version, 'all' );
	}

	/**
	 * Register the JavaScript for the admin area.
	 *
	 * @since    1.0.0
	 */
	public function enqueue_scripts() {
		wp_enqueue_script( $this->plugin_name, BUNDLE_WP_URL . 'admin/js/bundle-wp-admin.js', array( 'jquery' ), $this->version, false );
	}

	/**
	 * Block Copy Paste module instance.
	 *
	 * @since    1.0.0
	 * @access   private
	 * @var      Bundle_WP_Block_Copy_Paste    $block_copy_paste    Block Copy Paste module instance.
	 */
	private $block_copy_paste;
	
	/**
	 * No Search Engine module instance.
	 *
	 * @since    1.0.0
	 * @access   private
	 * @var      Bundle_WP_No_Search_Engine    $no_search_engine    No Search Engine module instance.
	 */
	private $no_search_engine;
	
	/**
	 * WebView to App module instance.
	 *
	 * @since    1.0.0
	 * @access   private
	 * @var      Bundle_WP_WebView_To_App    $webview_to_app    WebView to App module instance.
	 */
	private $webview_to_app;
	
	/**
	 * Add menu items for the plugin in WordPress admin.
	 *
	 * @since    1.0.0
	 */
	public function add_plugin_admin_menu() {
		// Top level menu
		add_menu_page(
			__( 'Bundle WP', 'bundle-wp' ),
			__( 'Bundle WP', 'bundle-wp' ),
			'manage_options',
			'bundle-wp',
			array( $this, 'display_plugin_dashboard' ),
			'dashicons-admin-plugins',
			100
		);
		
		// Initialize modules
		$this->block_copy_paste = new Bundle_WP_Block_Copy_Paste( $this->plugin_name, $this->version );
		$this->no_search_engine = new Bundle_WP_No_Search_Engine( $this->plugin_name, $this->version );
		$this->webview_to_app = new Bundle_WP_WebView_To_App( $this->plugin_name, $this->version );
		
		// Add module submenus
		add_submenu_page(
			'bundle-wp',
			__( 'Block Copy & Paste', 'bundle-wp' ),
			__( 'Block Copy & Paste', 'bundle-wp' ),
			'manage_options',
			'bundle-wp-block-copy-paste',
			array( $this->block_copy_paste, 'display_admin_page' )
		);
		
		add_submenu_page(
			'bundle-wp',
			__( 'No Search Engine', 'bundle-wp' ),
			__( 'No Search Engine', 'bundle-wp' ),
			'manage_options',
			'bundle-wp-no-search-engine',
			array( $this->no_search_engine, 'display_admin_page' )
		);
		
		add_submenu_page(
			'bundle-wp',
			__( 'WebView to App', 'bundle-wp' ),
			__( 'WebView to App', 'bundle-wp' ),
			'manage_options',
			'bundle-wp-webview-to-app',
			array( $this->webview_to_app, 'display_admin_page' )
		);
	}
	
	/**
	 * Display the plugin dashboard page.
	 *
	 * @since    1.0.0
	 */
	public function display_plugin_dashboard() {
		require_once BUNDLE_WP_PATH . 'admin/partials/bundle-wp-admin-display.php';
	}
}